<?php
declare(strict_types=1);

namespace Opencart\Admin\Controller\Extension\StsnDevSeoUrlFix\Other;

require_once DIR_EXTENSION . 'stsn_dev_seo_url_fix/helper.php';
use Opencart\Extension\StsnDevSeoUrlFix\StsnDevSeoUrlFixHelper as ext;

class StsnDevSeoUrlFix extends \Opencart\System\Engine\Controller
{
    const SLUGS = [
        'ru' => ['lang' => '/ru', 'ext' => 'seo-url-fix'],
        'uk' => ['lang' => '', 'ext' => 'seo-url-fix-ua'],
        'en' => ['lang' => '/en', 'ext' => 'seo-url-fix-en'],
    ];

    private ext $helper;
    private object $extModelsClass;
    private readonly string $separator;
    private readonly array $extInstallData;
    private readonly array $storesData;
    private readonly array $languagesInfo;

    public function __construct(\Opencart\System\Engine\Registry $registry)
    {
        parent::__construct($registry);

        $this->helper = new ext($this);

        $this->separator = version_compare(VERSION, '4.0.2.0', '>=') ? '.' : '|';

        $this->load->model(ext::FULL_PATH);
        $this->extModelsClass = $this->{ext::MODEL_ALIAS};

        $this->extInstallData = $this->extModelsClass->getExtensionInstallData(ext::CODE);

        $default_store_data = [
            0 => [
                'store_id' => 0,
                'name'     => $this->config->get('config_name'),
                'url'      => defined('HTTPS_CATALOG') ? HTTPS_CATALOG : (defined('HTTP_CATALOG') ? HTTP_CATALOG : ''),
            ]
        ];
        $this->load->model('setting/store');
        $other_stores_data = $this->model_setting_store->getStores();
        $other_stores_indexed = array_combine(
            array_column($other_stores_data, 'store_id'),
            $other_stores_data
        );
        $this->storesData = $default_store_data + $other_stores_indexed;
        // $this->log->write('storesData: ' . print_r($this->storesData, true));
        $this->languagesInfo = $this->model_localisation_language->getLanguages();
        // $this->log->write('languagesInfo: ' . print_r($this->languagesInfo, true));
    }

    private function formatLocalizedDate(\DateTime $date): string {
        $timezone = $this->config->get('config_timezone') ?? 'UTC';
        $date->setTimezone(new \DateTimeZone($timezone));

        // Получаем формат даты из текущего языка
        $format = $this->language->get('datetime_format') ?? 'Y-m-d H:i:s';

        return $date->format($format);
    }

    /**
     * Проверяет системные требования (наличие ionCube Loader и версия PHP).
     *
     * @return array {
     *     Результат проверки системы.
     *
     *     @type bool   'is_valid' Флаг валидности системы (true — если ошибок нет).
     *     @type string 'error'    Сообщение с описанием ошибок (пустая строка, если ошибок нет).
     * }
     */
    private function check_system(): array
    {
        $errors = [];

        if (!extension_loaded('ionCube Loader')) {
            $errors[] = $this->language->get('error_ion_cube');
        }

        if (version_compare(PHP_VERSION, ext::MIN_PHP_VERSION, '<')) {
            $errors[] = sprintf(
                $this->language->get('error_php'),
                ext::MIN_PHP_VERSION,
                PHP_VERSION
            );
        }

        return [
            'is_valid' => empty($errors),
            'error'    => implode(' ', $errors)
        ];
    }


    /**
     * @param array<int, string> $licenses  store_id => license_key
     * @param bool is_ajax
     * @return array
     */
    private function checkLicenses(array $licenses, bool $isAjax): array
    {
        // Инициализация массивов и флагов
        $errors = [];
        $messages = [];
        $storesOutput = [];
        $isMainStoreLicensed = false;

        foreach ($licenses as $storeId => $license) {
            $storeData = $this->storesData[$storeId];
            $host = parse_url($storeData['url'], PHP_URL_HOST) ?: '';

            // В режиме не-AJAX: инициализация структуры вывода
            if (!$isAjax) {
                $storesOutput['stores'][$storeId] = [
                    'field_label' => $storeData['name'],
                    'license'     => $license,
                ];
            }

            // Локальный хост или домен .local: лицензия не требуется
            if (str_contains($host, 'localhost') || str_ends_with($host, '.local')) {
                $message = $this->language->get('text_licensed_not_needed');
                if ($isAjax) {
                    $messages["store-license-$storeId"] = [
                        'text'       => $message,
                        'hide_input' => true,
                    ];
                } else {
                    $storesOutput['stores'][$storeId]['message']    = $message;
                    $storesOutput['stores'][$storeId]['hide_input'] = true;
                }

                // Отметить лицензию основного магазина, если идентификатор равен 0
                if ($storeId === 0) {
                    $isMainStoreLicensed = true;
                }
                continue;
            }

            // Если лицензия не передана и магазин основной: ошибка
            if (!$license) {
                $error = $storeId === 0
                    ? $this->language->get('text_main_domain_required')
                    : $this->language->get('license_empty');

                if ($isAjax) {
                    $errors["store-license-$storeId"] = $error;
                    if ($storeId === 0) {
                        $errors['warning'] = $this->language->get('error_form');
                    }
                } else {
                    $storesOutput['stores'][$storeId]['error'] = $error;
                }
                continue;
            }

            // Проверка длины лицензии: должна быть строкой длины 64 (или пустая)
            if (!is_string($license) || !in_array(strlen($license), [0, 64], true)) {
                $error = $this->language->get('str_len_licence_error');
                if ($isAjax) {
                    $errors["store-license-$storeId"] = $error;
                } else {
                    $storesOutput['stores'][$storeId]['error'] = $error;
                }
                continue;
            }

            $secretKeyFile = DIR_EXTENSION . 'stsn_dev_seo_url_fix/secret_key.php';
            if (file_exists($secretKeyFile)) {
                try {
                    include_once $secretKeyFile;
                    // Проверка статической лицензии (SHA-256 с секретным ключом, предварительно удеждаемся что secret_key.php раскодировался)
                    if (\Opencart\Extension\StsnDevSeoUrlFix\SecretKey::verifyPermanentLicense($license, $host)) {
                        $message = $this->language->get('text_licensed');
                        if ($isAjax) {
                            $messages["store-license-$storeId"] = [
                                'text'       => $message,
                                'hide_input' => true,
                            ];
                        } else {
                            $storesOutput['stores'][$storeId]['message'] = $message;
                            // Скрыть поле ввода после успешной лицензии
                            $storesOutput['stores'][$storeId]['hide_input'] = true;
                        }

                        // Отметить лицензию основного магазина, если идентификатор равен 0
                        if ($storeId === 0) {
                            $isMainStoreLicensed = true;
                        }
                        continue;
                    }
                } catch (\Throwable $e) {
                    $message = 'License check failed: ' . $e->getMessage();
                }
            }

            // Удалённая проверка лицензии/триала
            $extensionCheckUrl = $this->extInstallData['link'] . '/extensions/check-trial/';
            $requestData = [
                'license'        => $license,
                'extension_code' => ext::CODE,
            ];

            $curl = curl_init($extensionCheckUrl);
            curl_setopt_array($curl, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST           => true,
                CURLOPT_POSTFIELDS     => http_build_query($requestData),
                CURLOPT_TIMEOUT        => 5,
                CURLOPT_HTTPHEADER     => [
                    'X-Requested-With: XMLHttpRequest',
                    'Accept: application/json',
                    'User-Agent: Mozilla/5.0 (compatible; PHP-cURL/1.0)'
                ],
            ]);

            $response = curl_exec($curl);
            $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            curl_close($curl);

            if ($httpCode === 200 && $response !== false) {
                $serverJson = json_decode($response, true);
            } else {
                $serverJson = null;
            }

            if (is_array($serverJson) && isset($serverJson['check_result'])) {
                switch ($serverJson['check_result']) {
                    case 'not-found':
                        $error = $this->language->get('error_trial_not_found');
                        if ($isAjax) {
                            $errors["store-license-$storeId"] = $error;
                        } else {
                            $storesOutput['stores'][$storeId]['error'] = $error;
                        }
                        break;

                    case 'activated':
                        $expire_d_obj = new \DateTime($serverJson['expire']);
                        $message = $this->language->get('trial_activated') . $serverJson['expire'];
                        if ($isAjax) {
                            $messages["store-license-$storeId"] = ['text' => $message];
                        } else {
                            $storesOutput['stores'][$storeId]['message'] = $message;
                        }
                        // Отметить лицензию основного магазина, если идентификатор равен 0
                        if ($storeId === 0) {
                            $isMainStoreLicensed = true;
                        }
                        break;

                    case 'found':
                        $expire = new \DateTime($serverJson['expire']);
                        $now = new \DateTime('now', new \DateTimeZone('UTC'));

                        if ($now < $expire) {
                            $message = $this->language->get('use_trial_valid_to') . $this->formatLocalizedDate($expire);
                            if ($isAjax) {
                                $messages["store-license-$storeId"] = ['text' => $message];
                            } else {
                                $storesOutput['stores'][$storeId]['message'] = $message;
                            }
                        } else {
                            $error = $this->language->get('error_trial_expire');
                            if ($isAjax) {
                                $errors["store-license-$storeId"] = $error;
                            } else {
                                $storesOutput['stores'][$storeId]['error'] = $error;
                            }
                        }
                        // Отметить лицензию основного магазина, если идентификатор равен 0
                        if ($storeId === 0) {
                            $isMainStoreLicensed = true;
                        }
                        break;

                    default:
                        $error = $this->language->get('error_invalid_server_response');
                        if ($isAjax) {
                            $errors["store-license-$storeId"] = $error;
                        } else {
                            $storesOutput['stores'][$storeId]['error'] = $error;
                        }
                        break;
                }
            } else {
                // Запрос не удался или ответ неверный
                $error = $this->language->get('permanent_license_wrong_server_unreachable');
                if ($isAjax) {
                    $errors["store-license-$storeId"] = $error;
                } else {
                    $storesOutput['stores'][$storeId]['error'] = $error;
                }
            }
        }

        // Возврат собранного результата
        if ($isAjax) {
            return [
                'error'                 => $errors,
                'messages'               => $messages,
                'is_main_store_licensed' => $isMainStoreLicensed,
            ];
        }

        $storesOutput['is_main_store_licensed'] = $isMainStoreLicensed;
        return $storesOutput;
    }

    /**
     * Получает последнюю доступную версию расширения с удалённого сервера.
     *
     * Делает POST-запрос к URL, указанному в $this->extInstallData['link'],
     * добавляя к нему путь 'extensions/get-last-version/' и передавая код расширения.
     *
     * @return string|null Возвращает строку с версией (например, '1.2.3') при успешном запросе,
     *                     или null в случае ошибки или недоступности данных.
     */
    private function getLastVersion(){
        $extensionCheckUrl = $this->extInstallData['link'] . '/extensions/get-last-version/';

        $requestData = [
            'extension_code' => ext::CODE,
            'language_code' => $this->language->get('code')
        ];

        $curl = curl_init($extensionCheckUrl);
        curl_setopt_array($curl, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => http_build_query($requestData),
            CURLOPT_TIMEOUT        => 5,
            CURLOPT_HTTPHEADER     => [
                'X-Requested-With: XMLHttpRequest',
                'Accept: application/json',
                'User-Agent: Mozilla/5.0 (compatible; PHP-cURL/1.0)'
            ],
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        if ($httpCode === 200 && $response !== false) {
            $serverJson = json_decode($response, true);
            return $serverJson;
        } else {
            $serverJson = null;
        }
        return $serverJson;
    }


    private function getMainControllersSeoData(): array {
        $this->load->language(ext::FULL_PATH);
        $controllersList = [
            [
                'route' => 'cms/blog',
                'description' => $this->language->get('controller_description_blog_main_page'),
                'example_keyword_value' => 'blog',
                'values_by_lang' => []
            ],
            [
                'route' => 'product/compare',
                'description' => $this->language->get('controller_description_product_comparison'),
                'example_keyword_value' => 'comparison',
                'values_by_lang' => []
            ],
            [
                'route' => 'product/manufacturer',
                'description' => $this->language->get('controller_description_brands'),
                'example_keyword_value' => 'brands',
                'values_by_lang' => []
            ],
            [
                'route' => 'product/search',
                'description' => $this->language->get('controller_description_search'),
                'example_keyword_value' => 'search',
                'values_by_lang' => []
            ],
            [
                'route' => 'product/special',
                'description' => $this->language->get('controller_description_special_offers'),
                'example_keyword_value' => 'special',
                'values_by_lang' => []
            ],
            [
                'route' => 'information/contact',
                'description' => $this->language->get('controller_description_contact_us'),
                'example_keyword_value' => 'contacts',
                'values_by_lang' => []
            ],
            [
                'route' => 'information/sitemap',
                'description' => $this->language->get('controller_description_site_map'),
                'example_keyword_value' => 'sitemap',
                'values_by_lang' => []
            ],
            [
                'route' => 'checkout/cart',
                'description' => $this->language->get('controller_description_cart'),
                'example_keyword_value' => 'cart',
                'values_by_lang' => []
            ],
            [
                'route' => 'checkout/checkout',
                'description' => $this->language->get('controller_description_checkout'),
                'example_keyword_value' => 'checkout',
                'values_by_lang' => []
            ],
            [
                'route' => 'checkout/success',
                'description' => $this->language->get('controller_description_order_success'),
                'example_keyword_value' => 'order-success',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/account',
                'description' => $this->language->get('controller_description_account'),
                'example_keyword_value' => 'account',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/login',
                'description' => $this->language->get('controller_description_login'),
                'example_keyword_value' => 'login',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/logout',
                'description' => $this->language->get('controller_description_logout'),
                'example_keyword_value' => 'logout',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/affiliate',
                'description' => $this->language->get('controller_description_affiliate'),
                'example_keyword_value' => 'affiliate',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/order',
                'description' => $this->language->get('controller_description_orders'),
                'example_keyword_value' => 'orders',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/returns.add',
                'description' => $this->language->get('controller_description_return_add'),
                'example_keyword_value' => 'add-return',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/returns',
                'description' => $this->language->get('controller_description_returns'),
                'example_keyword_value' => 'returns',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/wishlist',
                'description' => $this->language->get('controller_description_wishlist'),
                'example_keyword_value' => 'wishlist',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/newsletter',
                'description' => $this->language->get('controller_description_newsletter_subscription'),
                'example_keyword_value' => 'newsletter-subscriptions',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/edit',
                'description' => $this->language->get('controller_description_personal_details'),
                'example_keyword_value' => 'personal-details',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/password',
                'description' => $this->language->get('controller_description_change_password'),
                'example_keyword_value' => 'change-password',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/payment_method',
                'description' => $this->language->get('controller_description_payment_methods'),
                'example_keyword_value' => 'payment-methods',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/address',
                'description' => $this->language->get('controller_description_address_entries'),
                'example_keyword_value' => 'address',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/download',
                'description' => $this->language->get('controller_description_downloads'),
                'example_keyword_value' => 'downloads',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/subscription',
                'description' => $this->language->get('controller_description_subscriptions'),
                'example_keyword_value' => 'subscriptions',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/reward',
                'description' => $this->language->get('controller_description_reward'),
                'example_keyword_value' => 'rewards',
                'values_by_lang' => []
            ],
            [
                'route' => 'account/transaction',
                'description' => $this->language->get('controller_description_transactions'),
                'example_keyword_value' => 'transactions',
                'values_by_lang' => []
            ],
        ];

        foreach ($controllersList as &$controller) {
            foreach ($this->languagesInfo as $language) {
                $language_id = $language['language_id'];
                $language_code = $language['code'];

                $controller['values_by_lang'][$language_id] = [
                    'lang_code' => $language_code,
                    'stores' => []
                ];

                foreach ($this->storesData as $store_id => $store) {
                    $store_name = $store['name'] ?? '';

                    $keyword = $this->extModelsClass->getKeywordByQuery($controller['route'], $store_id, $language_id);

                    $controller['values_by_lang'][$language_id]['stores'][$store_id] = [
                        'store_name' => $store_name,
                        'lang_keyword' => $keyword
                    ];
                }
            }
        }
        unset($controller);

        return $controllersList;
    }

    protected function makeUrl(string $route, array $params = []): string {
        $user_token = $this->session->data['user_token'];
        return $this->url->link($route, http_build_query(['user_token' => $user_token] + $params));
    }


    /**
     * @return null
     */
    public function index(): void {
        $this->load->language(ext::FULL_PATH);
        $this->document->setTitle($this->extInstallData['name']);

        $extDevSiteSlugs = self::SLUGS[$this->language->get('code')] ?? self::SLUGS['en'];
        $current_version = $this->extInstallData['version'];
        $last_version = $this->getLastVersion();

        $update_version = null;

        if ($last_version !== null && version_compare($last_version['version'], $current_version, '>')) {
            $update_version = $last_version;
        }
        $check_system_data = $this->check_system();

        $data = [
            'breadcrumbs' => [
                [
                    'text' => $this->language->get('text_home'),
                    'href' => $this->makeUrl('common/dashboard'),
                ],
                [
                    'text' => $this->language->get('text_extension'),
                    'href' => $this->makeUrl('marketplace/extension', ['type' => ext::TYPE]),
                ],
                [
                    'text' => $this->language->get('heading_title'),
                    'href' => $this->makeUrl(ext::FULL_PATH),
                ],
            ],

            'back' => $this->makeUrl('marketplace/extension', ['type' => ext::TYPE]),
            'text_edit' => sprintf($this->language->get('text_edit'), $this->extInstallData['name']),

            'header' => $this->load->controller('common/header'),
            'column_left' => $this->load->controller('common/column_left'),
            'footer' => $this->load->controller('common/footer'),

            'logo' => HTTP_CATALOG . 'extension/' . ext::CODE . '/admin/view/image/stsn-dev.svg',
            'extension_name' => $this->extInstallData['name'],
            'extension_home_page'=>  $this->extInstallData['link'] . $extDevSiteSlugs['lang'] . '/extensions/' . $extDevSiteSlugs['ext'],
            'extension_version' => $current_version,
            'update_version' => $update_version,

            'check_system_data' => $check_system_data,
            'export_url' => $this->makeUrl(ext::FULL_PATH . $this->separator . 'export'),
            'import_url' => $this->makeUrl(ext::FULL_PATH . $this->separator . 'import'),
        ];

        if (isset($this->session->data['success'])) {
            $data['success'] = $this->session->data['success'];
            unset($this->session->data['success']);
        } else {
            $data['success'] = '';
        }

        if (isset($this->session->data['error'])) {
            $data['error'] = $this->session->data['error'];
            unset($this->session->data['error']);
        } else {
            $data['error'] = '';
        }

        // Добавим только если check_system_data['is_valid'] === true
        if (!empty($check_system_data['is_valid'])) {
            // Конфигурационные ключи
            $config_keys = [
                'status',
                'hreflang_status',
                'xdefault_lang_code',
                'flat_mode',
                'offset_positions_default',
                'offset_positions_custom',
                'default_routes_to_skip_restore',
                'custom_routes_to_skip_restore'
            ];
            foreach ($config_keys as $key) {
                $value = $this->helper->getExtConfigValue($key);
                $data[$key] = $value;
                $configData[$key] = $value;
            }

            // $licenses_data для отрисовки полей дицензий
            $savedLicenses = [];
            foreach ($this->storesData as $store_id => $store) {
                $savedLicenses[$store_id] = $this->model_setting_setting->getValue(ext::CONF_PREFIX . '_license', $store_id);
            }
            $checked_licenses_data = $this->checkLicenses($savedLicenses, false);

            $data += [
                'save' => $this->makeUrl(ext::FULL_PATH . $this->separator . 'save'),
                'entry_status' => sprintf($this->language->get('entry_status'), $this->extInstallData['name']),
                'rendered_language_form_field' => $this->renderLanguageFormFields(
                    (bool)$configData['status'],
                ),
                'get_ajax_language_rerendered_form_fields_link' => $this->makeUrl(ext::FULL_PATH . $this->separator . 'setOutputRenderedLanguageFormFields'),
                'licenses_data' => $checked_licenses_data,
                'stores' => $this->storesData,
                'stock_seo_url_status' => $this->config->get('config_seo_url'),
                'all_languages_full_codes' => array_keys($this->languagesInfo),
                'main_conrollers_seo_data' => $this->getMainControllersSeoData()
            ];
        }
        $this->response->setOutput($this->load->view(ext::FULL_PATH, $data));
    }

    private function renderLanguageFormFields(bool $extension_status): string {
        $this->load->model('localisation/language');
        $total_languages = count($this->languagesInfo);
        $data['extension_status'] = $extension_status;
        $data['default_catalog_language'] = $this->config->get(
            version_compare(VERSION,'4.1.0.0','>=') ? 'config_language_catalog' : 'config_language'
        );
        $data['seo_language_registration_status_by_stores'] = [];

        foreach ($this->storesData as $store_id => $store_data) {
            $data['seo_language_registration_status_by_stores'][$store_id] = [
                'store_name' => $store_data['name'],
                'registered_language_codes' => [],
                'conflict_or_not_registered_language_codes' => [],
            ];

            foreach ($this->languagesInfo as $languageInfo) {
                $language_code = $languageInfo['code'];

                $valid_keyword = $this->extModelsClass->checkLangSeoRecords($store_id, $language_code, $total_languages);

                $entry = [
                    'language_name' => $languageInfo['name'],
                    'language_code' => $language_code,
                    'language_status' => $languageInfo['status'],
                ];

                if (is_string($valid_keyword) && $valid_keyword !== '') {
                    $entry['keyword'] = $valid_keyword;
                    $data['seo_language_registration_status_by_stores'][$store_id]['registered_language_codes'][] = $entry;
                } else {
                    $data['seo_language_registration_status_by_stores'][$store_id]['conflict_or_not_registered_language_codes'][] = $entry;
                }
            }
        }
        return $this->load->view(ext::FILESYSTEM_PATH . 'language_fields', $data);
    }

    public function setOutputRenderedLanguageFormFields(): void {
        $this->load->language(ext::FULL_PATH);
        $this->response->setOutput($this->renderLanguageFormFields(true));
    }

    public function save(): void {
        $this->load->language(ext::FULL_PATH);
        $this->load->model('setting/startup');

        $replaceFormValues = [];

        if ($this->user->hasPermission('modify', ext::FULL_PATH)) {
            if (isset($this->request->post['store_licenses'])) {
                $JsonResponseData = $this->checkLicenses($this->request->post['store_licenses'], true);
                if (isset($this->request->post['languages'])) {
                    foreach (array_keys($this->storesData) as $store_id ) {
                        $registered_values = array_values($this->request->post['languages'][$store_id]['registered_languages'] ?? []);
                        $conflict_or_not_registered_languages = array_values($this->request->post['languages'][$store_id]['conflict_or_not_registered_languages'] ?? []);

                        $all_values = array_merge($registered_values, $conflict_or_not_registered_languages);
                        if (isset($this->request->post['languages'][$store_id]['registered_languages'])) {
                            foreach ($this->request->post['languages'][$store_id]['registered_languages'] as $code => $key) {
                                $decode_key = html_entity_decode($key, ENT_QUOTES, 'UTF-8'); // очзаем от HTML-сущностей типа: &amp;, ', &quot; и т.д.
                                if (preg_match('/^[a-z]+$/u', $decode_key)) {
                                    if (count(array_filter($all_values, fn($item) => $item === $key)) > 1) { // проверяем уникальность:
                                        $JsonResponseData['error']['language_' . $store_id . '_' . $code] = $this->language->get('error_value_not_unique');
                                        $JsonResponseData['error']['warning'] = $this->language->get('error_form');
                                    } else {
                                        $replaceFormValues[sprintf('registered_languages[%s][%s]', $store_id, $code)] = $decode_key; // значения полей формы заменяем на очищенные
                                    }
                                } else {
                                    preg_match_all('/[^a-z]/u', $decode_key, $matches_key);
                                    $JsonResponseData['error']['language_' . $store_id . '_' . $code] = $this->language->get('error_invalid_characters') . implode(' ', $matches_key[0]);
                                    $JsonResponseData['error']['warning'] = $this->language->get('error_form');
                                }
                            }
                        }
                        if (isset($this->request->post['languages'][$store_id]['conflict_or_not_registered_languages'])) {
                            foreach ($this->request->post['languages'][$store_id]['conflict_or_not_registered_languages'] as $code => $key) {
                                $decode_key = html_entity_decode($key, ENT_QUOTES, 'UTF-8');
                                if (preg_match('/^[a-z]+$/u', $decode_key)) {
                                    if (count(array_filter($all_values, fn($item) => $item === $key)) > 1) {
                                        $JsonResponseData['error']['language_' . $store_id . '_' . $code] = $this->language->get('error_value_not_unique');
                                        $JsonResponseData['error']['warning'] = $this->language->get('error_form');
                                    } else {
                                        $replaceFormValues[sprintf('conflict_or_not_registered_languages[%s][%s]', $store_id, $code)] = $decode_key;
                                    }
                                } else {
                                    preg_match_all('/[^a-z]/u', $decode_key, $matches_key);
                                    $JsonResponseData['error']['language_' . $store_id . '_' . $code] = $this->language->get('error_invalid_characters') . implode(' ', $matches_key[0]);
                                    $JsonResponseData['error']['warning'] = $this->language->get('error_form');
                                }
                            }
                        }
                    }
                }

                if (isset($this->request->post['custom_offsets'])) {
                    $formOffsetsKeyValues = array_column($this->request->post['custom_offsets'] ?? [], 'key');
                    $defaultOffsetsKeyValues = array_keys($this->helper->getExtConfigValue('offset_positions_default'));
                    $allOffsetsKeyValues = array_merge($defaultOffsetsKeyValues, $formOffsetsKeyValues);
                    foreach ($this->request->post['custom_offsets'] as $index => $offsetData) {
                        $decode_key = html_entity_decode($offsetData['key'], ENT_QUOTES, 'UTF-8');
                        if (preg_match('/^[a-z_]+$/u', $decode_key)) {
                            if (count(array_filter($allOffsetsKeyValues, fn($item) => $item === $offsetData['key'])) > 1) {
                                $JsonResponseData['error']['offset_' . $index] = $this->language->get('error_key_not_unique');
                                $JsonResponseData['error']['warning'] = $this->language->get('error_form');
                            } else {
                                $replaceFormValues[sprintf('custom_offsets[%s][key]', $index)] = $decode_key;
                            }
                        } else {
                            preg_match_all('/[^a-z_]/u', $decode_key, $matches_key);
                            $JsonResponseData['error']['offset_' . $index] = $this->language->get('error_invalid_characters') . implode(' ', $matches_key[0]);
                            $JsonResponseData['error']['warning'] = $this->language->get('error_form');
                        }
                    }
                }

                if (isset($this->request->post['custom_routes'])) {
                    $routesFormData = $this->request->post['custom_routes'];
                    $routesDefaultData = $this->helper->getExtConfigValue('default_routes_to_skip_restore');

                    $allRoutesValues = array_merge(array_keys($routesDefaultData), array_column($routesFormData, 'route'));
                    $allObjectIdValues = array_merge(array_values($routesDefaultData), array_column($routesFormData, 'object_id'));

                    foreach ($routesFormData as $index => $routeData) {
                        $decode_route = html_entity_decode($routeData['route'], ENT_QUOTES, 'UTF-8');
                        if (preg_match('/^[a-z_\/.]+$/u', $decode_route)) {
                            if (count(array_filter($allRoutesValues, fn($item) => $item === $routeData['route'])) > 1) {
                                $JsonResponseData['error']['route_' . $index] = $this->language->get('error_value_not_unique');
                                $JsonResponseData['error']['warning'] = $this->language->get('error_form');
                            } else {
                                $replaceFormValues[sprintf('custom_routes[%s][route]', $index)] = $decode_route;
                            }
                        } else {
                            preg_match_all('/[^a-z_\/.]/u', $routeData['route'], $matches_route);
                            $JsonResponseData['error']['route_' . $index] = $this->language->get('error_invalid_characters') . implode(' ', $matches_route[0]);
                            $JsonResponseData['error']['warning'] = $this->language->get('error_form');
                        }
                        $decode_object_id = html_entity_decode($routeData['object_id'], ENT_QUOTES, 'UTF-8');
                        if (preg_match('/^[a-z_]+$/u', $decode_object_id)) {
                            if (count(array_filter($allObjectIdValues, fn($item) => $item === $routeData['object_id'])) > 1) {
                                $JsonResponseData['error']['object_id_' . $index] = $this->language->get('error_value_not_unique');
                                $JsonResponseData['error']['warning'] = $this->language->get('error_form');
                            } else {
                                $replaceFormValues[sprintf('custom_routes[%s][object_id]', $index)] = $decode_object_id;
                            }
                        } else {
                            preg_match_all('/[^a-z_]/u', $decode_object_id, $matches_object_id);
                            $JsonResponseData['error']['object_id_' . $index] = $this->language->get('error_invalid_characters') . implode(' ', $matches_object_id[0]);
                            $JsonResponseData['error']['warning'] = $this->language->get('error_form');
                        }
                    }
                }

                if (isset($this->request->post['main_controller_keywords'])) {

                    $allKeywordsByLangStore = [];

                    // 1. Сначала собираем все keywords по языкам и магазинам
                    foreach ($this->request->post['main_controller_keywords'] as $route_name => $routeKeywordsByLang) {
                        foreach ($routeKeywordsByLang as $lang_id => $routeKeywordsByStore) {
                            foreach ($routeKeywordsByStore as $store_id => $keyword) {
                                $trimmedKeyword = trim($keyword);

                                // Собираем для проверки уникальности в рамках языка и магазина
                                $allKeywordsByLangStore[$lang_id][$store_id][] = [
                                    'route'   => $route_name,
                                    'keyword' => $trimmedKeyword
                                ];
                            }
                        }
                    }

                    // 2. Теперь проверяем уникальность и формат
                    foreach ($this->request->post['main_controller_keywords'] as $route_name => $routeKeywordsByLang) {
//                        $this->log->write('------------- CHECK -' . $route_name . '----------------');
                        $safe_route_name = str_replace('/', '-', $route_name);

                        foreach ($routeKeywordsByLang as $lang_id => $routeKeywordsByStore) {
                            foreach ($routeKeywordsByStore as $store_id => $keyword) {

                                $keyword = trim($keyword);
                                $field_id = "main_controller_keyword-{$safe_route_name}-{$lang_id}-{$store_id}";

//                                $this->log->write("lang_id: {$lang_id}, store_id: {$store_id}, keyword: {$keyword}");

                                // 2.1 Проверка уникальности
                                $countSame = 0;
                                foreach ($allKeywordsByLangStore[$lang_id][$store_id] as $data) {
                                    if ($data['keyword'] === $keyword && $keyword !== '') {
                                        $countSame++;
                                    }
                                }

                                if ($countSame > 1) {
                                    $JsonResponseData['error'][$field_id] = $this->language->get('error_value_not_unique');
                                    $JsonResponseData['error']['warning'] = $this->language->get('error_form');
                                }

                                // 2.2 Проверка допустимых символов: только a-z и дефис
                                if ($keyword !== '' && !preg_match('/^[a-z\-]+$/u', $keyword)) {
                                    preg_match_all('/[^a-z\-]/u', $keyword, $bad_chars);
                                    $JsonResponseData['error'][$field_id] =
                                        $this->language->get('error_invalid_characters') . implode(' ', $bad_chars[0]);
                                    $JsonResponseData['error']['warning'] = $this->language->get('error_form');
                                } else if ($keyword !== '') {
                                    $found = $this->extModelsClass->getRouteByKeyword($lang_id, $store_id, $keyword);

                                    if ($found && $found['value'] !== $route_name) {
                                        $JsonResponseData['error'][$field_id] =
                                            sprintf(
                                                $this->language->get('error_keyword_used_by_another'),
                                                $found['value']
                                            );
                                        $JsonResponseData['error']['warning'] = $this->language->get('error_form');
                                    }
                                }
                            }
                        }
                    }
                }

            // ====================================== SAVE =============================================
                $startup_id = $this->model_setting_startup->getStartupByCode(ext::CODE)['startup_id'];
                if (empty($JsonResponseData['error']['warning'])) {
                    foreach ($this->request->post['store_licenses'] as $store_id => $license) {
                        $this->helper->editExtConfigValue('license', $license, $store_id);
                    }

                    if (isset($this->request->post['extension_status']) && $this->request->post['extension_status'] === '1') {
                        $this->helper->editExtConfigValue('flat_mode', $this->request->post['flat_mode']);
                        $this->helper->editExtConfigValue('hreflang_status', $this->request->post['hreflang_status']);
                        $this->helper->editExtConfigValue('xdefault_lang_code', $this->request->post['xdefault_lang_code']);
                        $this->helper->editExtConfigValue('offset_positions_default', $this->request->post['default_offsets']);

                        $offsetPositions = isset($this->request->post['custom_offsets']) && is_array($this->request->post['custom_offsets'])
                            ? array_combine(
                                $formOffsetsKeyValues,
                                array_column($this->request->post['custom_offsets'], 'offset')
                            )
                            : [];
                        $this->helper->editExtConfigValue('offset_positions_custom', $offsetPositions);

                        $routesSkipRestore = isset($this->request->post['custom_routes']) && is_array($this->request->post['custom_routes'])
                            ? array_combine(
                                array_column($this->request->post['custom_routes'], 'object_id'),
                                array_column($this->request->post['custom_routes'], 'route')
                            )
                            : [];
                        $this->helper->editExtConfigValue('custom_routes_to_skip_restore', $routesSkipRestore);

                        $this->helper->editExtConfigValue('status', 1);
                        $this->model_setting_startup->editStatus($startup_id, true);
                        $this->model_setting_setting->editValue('config', 'config_seo_url', 0); // Выключаем стандартный SEO URL

                        $this->extModelsClass->updateLangSeoUrls($this->request->post['languages']);
                        if (isset($this->request->post['main_controller_keywords'])) {
                            foreach ($this->request->post['main_controller_keywords'] as $route_name => $routeKeywordsByLang) {
//                                $this->log->write('------------ SAVE --' . $route_name . '----------------');
                                foreach ($routeKeywordsByLang as $lang_id => $routeKeywordsByStore) {
                                    foreach ($routeKeywordsByStore as $store_id => $keyword) {
//                                        $this->log->write('lang:_id: ' . $lang_id . ', store_id:' . $store_id . ', keyword: ' . $keyword);
                                        $this->extModelsClass->saveMainControllersSeoUrls($route_name, (int)$store_id, (int)$lang_id, trim($keyword));
                                    }
                                }
                            }
                        }
                    } else {
                        $this->model_setting_startup->editStatus($startup_id, false);
                        $this->helper->editExtConfigValue( 'status', 0);

                        // когда расщирение не активно задаем нужный stock SEO URL статус
                        $this->model_setting_setting->editValue(
                            'config',
                            'config_seo_url',
                            !empty($this->request->post['stock_seo_url_status']) ? 1 : 0
                        );

                    }

                    $JsonResponseData['success'] = sprintf($this->language->get('text_success'), $this->extInstallData['name']);
                    $JsonResponseData += $replaceFormValues;
                }

            } else {
                $JsonResponseData = [];
            }
        } else {
            $JsonResponseData['error']['warning'] = $this->language->get('error_permission');
        }
        $this->response->addHeader('Content-Type: application/json');
        $this->response->setOutput(json_encode($JsonResponseData));
    }

    public function install(): void {
        $this->load->model('setting/startup');
        if (!$this->model_setting_startup->getStartupByCode(ext::CODE)) {
            $this->model_setting_startup->addStartup(
                [
                    'code' => ext::CODE,
                    'action' => 'catalog/extension/' . ext::CODE .'/startup/seo_url',
                    'description' => $this->extInstallData['description'],
                    'sort_order' => 1,
                    'status' => false
                ]
            );
        }

        $this->load->model('setting/event');

        // Проверяем, существует ли уже событие с нужным кодом
        $existingEvent = $this->model_setting_event->getEventByCode('stsn_dev_seo_url_fix_hreflang');

        if (!$existingEvent) {
            $this->model_setting_event->addEvent([
                'code'       => 'stsn_dev_seo_url_fix_hreflang',
                'description'=> 'Add hreflang links to header',
                'trigger'    => 'catalog/view/common/header/after',
                'action'     => 'extension/stsn_dev_seo_url_fix/header_event' . $this->separator . 'insertHreflangLinks',
                'status'     => true,
                'sort_order' => 0
            ]);
        }

        $this->load->model('setting/setting');

        $offsetPositions = [
            'product_id' => 5,
            'article_id' => 5,
        ];

        $defaultRoutes = [
            'path' => 'product/category',
            'product_id' => 'product/product',
            'information_id' => 'information/information',
            'manufacturer_id' => 'product/manufacturer.info',
            'topic_id' => 'cms/blog',
            'article_id' => 'cms/blog.info',
        ];

        $this->model_setting_setting->editSetting(ext::CONF_PREFIX, [
            ext::CONF_PREFIX . '_status' => 0,
            ext::CONF_PREFIX . '_flat_mode' => 0,
            ext::CONF_PREFIX . '_hreflang_status' => 0,
            ext::CONF_PREFIX . '_xdefault_lang_code' => '',
            ext::CONF_PREFIX . '_offset_positions_default' => $offsetPositions,
            ext::CONF_PREFIX . '_offset_positions_custom' => [],
            ext::CONF_PREFIX . '_default_routes_to_skip_restore' => $defaultRoutes,
            ext::CONF_PREFIX . '_custom_routes_to_skip_restore' => [],
            ext::CONF_PREFIX . '_license' => '',
        ]);
    }

    public function uninstall(): void {
        $this->load->model('setting/startup');
        $this->model_setting_startup->deleteStartupByCode(ext::CODE);

        $this->load->model('setting/event');
        $this->model_setting_event->deleteEventByCode('stsn_dev_seo_url_fix_hreflang');
    }

    public function export() {
        if (!$this->user->hasPermission('modify', ext::FULL_PATH)) {
            return;
        }

        // Получаем все настройки по коду
        $settings = $this->db->query("SELECT * FROM `" . DB_PREFIX . "setting` WHERE `code` = '" . ext::CONF_PREFIX . "'")->rows;

        // Убираем setting_id из каждого элемента
        foreach ($settings as &$setting) {
            unset($setting['setting_id']);
        }
        unset($setting); // Очистка ссылки

        // Формируем имя файла с текущей датой и временем
        $datetime = date('Y-m-d_H-i-s');
        $filename = ext::CODE . '_settings_' . $datetime . '.json';

        header('Content-Type: application/json; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        echo json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit;
    }


    public function import() {
        if (!$this->user->hasPermission('modify', ext::FULL_PATH)) {
            return;
        }

        $this->load->language(ext::FULL_PATH);

        if (!empty($_FILES['settings']['tmp_name'])) {
            $filename = $_FILES['settings']['name'];

            if (strpos($filename, ext::CODE . '_settings_') === 0) {
                $json = json_decode(file_get_contents($_FILES['settings']['tmp_name']), true);

                if (is_array($json)) {
                    foreach ($json as $setting) {
                        $store_id = (int)$setting['store_id'];
                        $code = $this->db->escape($setting['code']);
                        $key = $this->db->escape($setting['key']);
                        $value = $this->db->escape($setting['value']);
                        $serialized = (int)$setting['serialized'];

                        $exists = $this->db->query("SELECT 1 FROM `" . DB_PREFIX . "setting` WHERE `store_id` = '$store_id' AND `code` = '$code' AND `key` = '$key' LIMIT 1");

                        if ($exists->num_rows) {
                            // Обновляем существующую запись
                            $this->db->query("UPDATE `" . DB_PREFIX . "setting` SET
                            `value` = '$value',
                            `serialized` = '$serialized'
                            WHERE `store_id` = '$store_id' AND `code` = '$code' AND `key` = '$key'
                        ");
                        } else {
                            // Вставляем новую
                            $this->db->query("INSERT INTO `" . DB_PREFIX . "setting` SET
                            `store_id` = '$store_id',
                            `code` = '$code',
                            `key` = '$key',
                            `value` = '$value',
                            `serialized` = '$serialized'
                        ");
                        }
                    }
                    $this->session->data['success'] = $this->language->get('success_import');
                } else {
                    $this->session->data['error'] = $this->language->get('error_file_json');
                }
            } else {
                $this->session->data['error'] = sprintf($this->language->get('error_file_prefix'), ext::CODE);;
            }
        } else {
            $this->session->data['error'] = $this->language->get('error_file_empty');
        }

        $url = $this->makeUrl(ext::FULL_PATH);

        if (!empty($this->session->data['error'])) {
            $this->log->write('error: ' . print_r($this->session->data['error'], true));
            $url .= '#tab-export-import';
        }

        $this->log->write('$url: ' . $url);

        $this->response->redirect($url);
    }


}
