<?php
declare(strict_types=1);

namespace Opencart\Extension\StsnDevAdminPlus;

class StsnDevHelper
{
    protected \Opencart\System\Engine\Controller $controller;

    public const CODE = 'stsn_dev_admin_plus';
    public const TYPE = 'other';
    public const MIN_PHP_VERSION = '8.2.0';
    public const FILESYSTEM_PATH = 'extension/' . self::CODE . '/' . self::TYPE . '/';
    public const FULL_PATH        = self::FILESYSTEM_PATH . self::CODE;
    public const MODEL_ALIAS      = 'model_extension_' . self::CODE . '_' . self::TYPE . '_' . self::CODE;
    public const CONF_PREFIX      = self::TYPE . '_' . self::CODE;

    private ?string $secretKey = null;
    public readonly string $separator;
    public readonly array $events;

    /**
     * Конструктор класса.
     *
     * @param \Opencart\System\Engine\Controller $controller  Контроллер, из которого вызывается хелпер.
     */
    public function __construct(\Opencart\System\Engine\Controller $controller) {
        $this->controller = $controller;
        $this->separator = version_compare(VERSION, '4.0.2.0', '>=') ? '.' : '|';

        $this->events = [
            [
                'code'    => self::CODE . '_view_before',
                'trigger' => 'admin/view/*/before',
                'action'  => 'extension/' . self::CODE . '/admin_event_actions' . $this->separator . 'view_event_before',
            ],
            [
                'code'    => self::CODE . '_view_after',
                'trigger' => 'admin/view/*/after',
                'action'  => 'extension/' . self::CODE . '/admin_event_actions' . $this->separator . 'view_event_after',
            ],
        ];

    }

    /**
     * Получает значение настройки по ключу с префиксом расширения.
     *
     * @param string $key  Короткий ключ настройки (без префикса).
     *
     * @return mixed       Значение настройки (тип зависит от хранимого значения).
     */
    public function getExtConfigValue(string $key): mixed {
        return $this->controller->config->get(self::CONF_PREFIX . '_' . $key);
    }

    /**
     * Сохраняет значение настройки для указанного ключа и магазина.
     *
     * @param string $key       Короткий ключ настройки (будет дополнен префиксом).
     * @param mixed  $value     Значение настройки.
     * @param int    $store_id  ID магазина (по умолчанию 0).
     *
     * @return void
     */
    public function editExtConfigValue(string $key, mixed $value, int $store_id = 0): void {
        $this->controller->model_setting_setting->editValue(
            self::CONF_PREFIX,
            self::CONF_PREFIX . '_' . $key,
            $value,
            $store_id
        );
    }

    public function save(): void {
        foreach ($this->controller->request->post as $key => $value) {
            $this->editExtConfigValue($key, $value);
        }
    }
}
