// ======================= Настройки =======================
const sortableTables = [
    '#module-column-left',
    '#module-content-top',
    '#module-content-bottom',
    '#module-column-right'
];

// проверяем, включен ли функционал
const draggableEnabled = adminPlusConfig?.config?.other_stsn_dev_admin_plus_draggable_layouts == 1;
const uniqueModulesEnabled = adminPlusConfig?.config?.other_stsn_dev_admin_plus_unique_module_selection == 1;

// ======================= Drag & Drop =======================
function updateSortOrder(table) {
    table.querySelectorAll('tbody tr').forEach((row, index) => {
        const input = row.querySelector('input[name$="[sort_order]"]');
        if (input) input.value = index;
    });
}

function addDragHandles(table) {
    if (!draggableEnabled) return;

    table.querySelectorAll('tbody tr').forEach(row => {
        const inputGroup = row.querySelector('.input-group');
        if (inputGroup && !row.querySelector('.sort-handle')) {
            const handle = document.createElement('i');
            handle.className = 'fa-solid fa-grip-vertical sort-handle me-2';
            handle.style.cursor = 'grab';
            handle.style.fontSize = '16px';
            handle.style.userSelect = 'none';

            inputGroup.style.display = 'flex';
            inputGroup.style.alignItems = 'center';

            inputGroup.prepend(handle);
        }
    });
}

function observeTable(table) {
    if (!draggableEnabled) return;

    const tbody = table.querySelector('tbody');
    if (!tbody) return;

    const observer = new MutationObserver(() => {
        addDragHandles(table);
        if (uniqueModulesEnabled) filterSelectOptions();
        updateSortOrder(table);
    });

    observer.observe(tbody, { childList: true });
}

// ======================= Фильтрация select =======================
function filterSelectOptions() {
    if (!uniqueModulesEnabled) return;

    const selects = Array.from(document.querySelectorAll(
        '#module-column-left select, #module-column-right select, #module-content-top select, #module-content-bottom select'
    ));

    const selectedValues = selects
        .map(s => s.value)
        .filter(v => v && v !== '');

    selects.forEach(select => {
        const currentValue = select.value;

        if (!select.dataset.fullOptions) {
            select.dataset.fullOptions = select.innerHTML;
        }

        const wrapper = document.createElement('select');
        wrapper.innerHTML = select.dataset.fullOptions;

        select.innerHTML = '';

        const emptyOption = document.createElement('option');
        emptyOption.value = '';
        emptyOption.text = '--------------';
        select.appendChild(emptyOption);

        Array.from(wrapper.children).forEach(child => {
            if (child.tagName === 'OPTION') {
                if (!selectedValues.includes(child.value) || child.value === currentValue) {
                    select.appendChild(child.cloneNode(true));
                }
            } else if (child.tagName === 'OPTGROUP') {
                const newGroup = document.createElement('optgroup');
                newGroup.label = child.label;
                Array.from(child.children).forEach(opt => {
                    if (!selectedValues.includes(opt.value) || opt.value === currentValue) {
                        newGroup.appendChild(opt.cloneNode(true));
                    }
                });
                if (newGroup.children.length) select.appendChild(newGroup);
            }
        });

        select.value = currentValue || '';
    });
}

// ======================= Инициализация таблиц =======================
sortableTables.forEach(selector => {
    const table = document.querySelector(selector);
    if (!table) return;

    addDragHandles(table);
    observeTable(table);

    if (draggableEnabled) {
        new Sortable(table.querySelector('tbody'), {
            animation: 150,
            handle: '.sort-handle',
            onEnd: () => updateSortOrder(table)
        });
    }

    table.addEventListener('change', e => {
        if (e.target.tagName === 'SELECT' && uniqueModulesEnabled) filterSelectOptions();
    });

    if (uniqueModulesEnabled) filterSelectOptions();
});

// ======================= Переопределение addModule =======================
if (typeof addModule === 'function') {
    const originalAddModule = addModule;
    addModule = function(type) {
        originalAddModule(type);

        setTimeout(() => {
            const table = document.querySelector('#module-' + type);
            addDragHandles(table);

            const newRow = table.querySelector('tbody tr:last-child');
            const select = newRow.querySelector('select');
            if (select) {
                if (!select.querySelector('option[value=""]')) {
                    const emptyOption = document.createElement('option');
                    emptyOption.value = '';
                    emptyOption.text = '--------------';
                    select.prepend(emptyOption);
                }
                select.value = '';
            }

            if (uniqueModulesEnabled) filterSelectOptions();
            updateSortOrder(table);
        }, 50);
    };
}

// ======================= Форма: игнорируем пустые selects и удаляем строки после сохранения =======================
$('#form-layout').on('submit', function() {
    // Отключаем пустые selects, чтобы не отправлялись
    $('#module-column-left select, #module-column-right select, #module-content-top select, #module-content-bottom select').each(function() {
        if (!this.value) $(this).prop('disabled', true);
    });

    // После отправки, удаляем строки с пустыми select
    setTimeout(() => {
        sortableTables.forEach(selector => {
            const table = document.querySelector(selector);
            if (!table) return;

            table.querySelectorAll('tbody tr').forEach(row => {
                const select = row.querySelector('select');
                if (select && !select.value) {
                    row.remove();
                }
            });

            updateSortOrder(table);
            if (uniqueModulesEnabled) filterSelectOptions();
        });
    }, 100);
});
