// ======================= Вспомогательные функции =======================

// Деактивируем/активируем SEO поля по выбранным магазинам
const updateSeoRows = () => {
    const storeCheckboxes = document.querySelectorAll(
        `input[name="${adminPlusConfig.routeConfig.storesFieldName}[]"]`
    );

    const hideStoresType = adminPlusConfig.config.other_stsn_dev_admin_plus_hide_stores_type || '0';

    storeCheckboxes.forEach(cb => {
        Object.values(adminPlusConfig.activeLanguages).forEach(langId => {
            const seoField = document.querySelector(
                `input[name="${adminPlusConfig.routeConfig.seoFieldNamePrefix}[${cb.value}][${langId}]"]`
            );
            if (!seoField) return;

            if (hideStoresType === '2') {
                // включаем/отключаем
                if (cb.checked) {
                    seoField.removeAttribute("disabled");
                } else {
                    seoField.setAttribute("disabled", "disabled");
                }
            } else if (hideStoresType === '1') {
                // обязательность для отмеченных
                if (cb.checked) {
                    seoField.setAttribute("required", "required");
                } else {
                    seoField.removeAttribute("required");
                }
            }
        });
    });
};


const generateSeo = () => {
    const storeCheckboxes = document.querySelectorAll(
        `input[name="${adminPlusConfig.routeConfig.storesFieldName}[]"]`
    );
    const namePrefix = adminPlusConfig?.routeConfig?.namePrefix || '';
    const nameSuffix = adminPlusConfig?.routeConfig?.nameSuffix || '';
    const seoFieldNamePrefix = adminPlusConfig?.routeConfig?.seoFieldNamePrefix || '';
    const languages = adminPlusConfig?.activeLanguages || {}; // объект {code: id}

    storeCheckboxes.forEach(cb => {
        if (!cb.checked && adminPlusConfig.config.other_stsn_dev_admin_plus_hide_disabled_stores_seo === '1') return;

        Object.entries(languages).forEach(([langCode, langId]) => {
            const nameFieldSelector = adminPlusConfig?.routeConfig?.namePrefix
                ? `input[name="${namePrefix}[${langId}][${nameSuffix}]"]`
                : `input[name="${nameSuffix}"]`;

            const seoFieldSelector = `input[name="${seoFieldNamePrefix}[${cb.value}][${langId}]"]`;

            const nameInput = document.querySelector(nameFieldSelector);
            const seoField = document.querySelector(seoFieldSelector);

            if (!nameInput || !seoField) {
                console.warn('Не найдено поле для генерации SEO:', { nameFieldSelector, seoFieldSelector });
                return;
            }

            const locale = ['ru', 'uk', 'en', 'de', 'vi'].includes(langCode.slice(0, 2))
                ? langCode.slice(0, 2)
                : undefined;

            const seoUrlValue = slugify(nameInput.value, {
                lower: true,
                strict: true,
                locale
            });

            seoField.dataset.expectedValue = seoUrlValue;

            const currentValue = (seoField.value || '').trim();
            const expectedValue = (seoUrlValue || '').trim();

            Array.from(seoField.parentNode.children)
                .filter(el => el.tagName === 'BUTTON' && el.classList.contains('seo-fix-btn'))
                .forEach(btn => btn.remove());

            if (currentValue !== expectedValue) {
                const btn = document.createElement('button');
                btn.type = 'button';
                btn.innerHTML = '🔁';
                btn.classList.add('seo-fix-btn');
                btn.style.marginRight = '5px';
                btn.style.border = 'none';
                btn.style.cursor = 'pointer';

                btn.addEventListener('click', () => {
                    seoField.value = expectedValue;
                    seoField.dispatchEvent(new Event('input', { bubbles: true }));
                    btn.remove();
                });

                seoField.parentNode.insertBefore(btn, seoField);
            }
        });
    });
};

const updateCatalogLinks = () => {
    const oldButtons = document.querySelector('#stsn-store-buttons');
    if (oldButtons) oldButtons.remove();

    const objectIdKey = adminPlusConfig?.routeConfig?.objectIdKey;
    const objectId = document.querySelector(`[name="${objectIdKey}"]`)?.value;
    const objectRoute = adminPlusConfig?.routeConfig?.objectRoute;
    const storesData = adminPlusConfig?.storesData;
    const urlParameter = adminPlusConfig?.routeConfig?.urlParameter;
    const languageParameter = adminPlusConfig?.routeConfig?.languageParameter;
    const checkedStoreIds = Array.from(document.querySelectorAll('#product-store input[type="checkbox"]:checked'))
        .map(cb => cb.value);

    const links = storesData
        .filter(store => checkedStoreIds.includes(store.store_id.toString()))
        .map(store => ({
            name: store.name,
            url: store.url + '/index.php?route=' + objectRoute + urlParameter + objectId + languageParameter
        }));

    if (!links.length) return;

    let buttonHtml = '<div id="stsn-store-buttons" class="d-inline-block btn-group-wrapper me-2">';
    if (links.length === 1) {
        const one = links[0];
        buttonHtml += '<a href="' + one.url + '" class="btn btn-success me-2" target="_blank" title="' + one.name + '"><i class="fa-solid fa-eye"></i></a>';
    } else {
        buttonHtml += '<div class="btn-group me-2">' +
            '<button type="button" class="btn btn-success dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">' +
            '<i class="fa-solid fa-eye"></i></button><ul class="dropdown-menu dropdown-menu-end">';
        links.forEach(lnk => {
            buttonHtml += '<li><a class="dropdown-item" target="_blank" href="' + lnk.url + '">' + lnk.name + '</a></li>';
        });
        buttonHtml += '</ul></div>';
    }
    buttonHtml += '</div>';

    const saveBtn = document.querySelector('button[type="submit"]');
    if (saveBtn) saveBtn.insertAdjacentHTML('beforebegin', buttonHtml);
};

const getCookie = (name) => {
    const value = `; ${document.cookie}`;
    const parts = value.split(`; ${name}=`);
    return parts.length === 2 ? parts.pop().split(';').shift() : null;
};

// ======================= Работа с изменениями полей =======================
const createChangeHandlerUniversal = (el) => {
    const isCheckbox = el.type === 'checkbox';
    const isSelect = el.tagName.toLowerCase() === 'select';

    el.dataset.originalValue = isCheckbox ? (el.checked ? "1" : "0") : el.value;

    const checkChanges = () => {
        let container = el.parentElement.querySelector('.change-container');

        const currentValue = isCheckbox ? (el.checked ? "1" : "0") : el.value;
        const originalValue = el.dataset.originalValue;

        if (currentValue !== originalValue) {
            if (!container) {
                container = document.createElement('div');
                container.className = 'change-container d-inline-flex align-items-center gap-1 mt-1';

                const pencil = document.createElement('span');
                pencil.textContent = '✏️';
                pencil.style.fontSize = '0.9em';

                const resetIcon = document.createElement('span');
                resetIcon.textContent = '⟳';
                resetIcon.style.cursor = 'pointer';
                resetIcon.style.fontSize = '0.9em';
                resetIcon.addEventListener('click', () => {
                    if (isCheckbox) el.checked = originalValue === "1";
                    else el.value = originalValue;

                    el.dispatchEvent(new Event(isCheckbox || isSelect ? 'change' : 'input', { bubbles: true }));

                    if (container) container.remove();
                    if (adminPlusConfig.config.other_stsn_dev_admin_plus_generate_seo === "1") generateSeo();
                });

                container.appendChild(pencil);
                container.appendChild(resetIcon);
                el.parentElement.appendChild(container);
            }
        } else if (container) {
            container.remove();
        }
    };

    el.addEventListener(isCheckbox || isSelect ? 'change' : 'input', checkChanges);
    checkChanges();
};

// ======================= Глобальный индикатор изменений =======================
const createGlobalChangeIndicator = (fields) => {
    const h1 = document.querySelector('h1');
    if (!h1) return;

    let indicator = h1.querySelector('.global-change-indicator');
    if (!indicator) {
        indicator = document.createElement('span');
        indicator.className = 'global-change-indicator';
        indicator.style.fontSize = '0.8em';
        indicator.style.marginLeft = '8px';
        indicator.style.display = 'none';
        indicator.style.alignItems = 'center';
        indicator.style.gap = '4px';

        const pencil = document.createElement('span');
        pencil.textContent = '✏️';
        pencil.style.fontSize = '0.9em';

        const countSpan = document.createElement('span');
        countSpan.className = 'global-change-count';
        countSpan.textContent = '(0)';

        const resetBtn = document.createElement('button');
        resetBtn.type = 'button';
        resetBtn.textContent = '⟳';
        resetBtn.style.cursor = 'pointer';
        resetBtn.style.padding = '0';
        resetBtn.style.border = 'none';
        resetBtn.style.background = 'transparent';
        resetBtn.style.fontSize = '0.9em';

        resetBtn.addEventListener('click', () => {
            fields.forEach(el => {
                const type = el.type;
                const isCheckbox = type === 'checkbox';
                const isSelect = el.tagName.toLowerCase() === 'select';
                const originalValue = el.dataset.originalValue;
                if (originalValue !== undefined) {
                    if (isCheckbox) el.checked = originalValue === "1";
                    else el.value = originalValue;

                    const eventType = isCheckbox || isSelect ? 'change' : 'input';
                    el.dispatchEvent(new Event(eventType, { bubbles: true }));
                }
            });
            updateCounter();
            if (adminPlusConfig.config.other_stsn_dev_admin_plus_generate_seo === "1") generateSeo();
        });

        indicator.appendChild(pencil);
        indicator.appendChild(countSpan);
        indicator.appendChild(resetBtn);
        h1.appendChild(indicator);
    }

    const updateCounter = () => {
        const changedFields = Array.from(fields).filter(el => {
            const type = el.type;
            const isCheckbox = type === 'checkbox';
            const isSelect = el.tagName.toLowerCase() === 'select';
            const originalValue = el.dataset.originalValue;
            if (originalValue === undefined) return false;
            const currentValue = isCheckbox ? (el.checked ? "1" : "0") : el.value;
            return currentValue !== originalValue;
        });

        const countSpan = indicator.querySelector('.global-change-count');
        if (countSpan) countSpan.textContent = `(${changedFields.length})`;
        indicator.style.display = changedFields.length > 0 ? 'inline-flex' : 'none';
    };

    fields.forEach(el => {
        const type = el.type;
        const isCheckbox = type === 'checkbox';
        const isSelect = el.tagName.toLowerCase() === 'select';
        const eventType = isCheckbox || isSelect ? 'change' : 'input';
        el.addEventListener(eventType, updateCounter);
    });

    updateCounter();
    return updateCounter;
};

// ======================= Сброс всех карандашей и глобального индикатора =======================
const resetChangeIndicators = (fields) => {
    fields.forEach(el => {
        const type = el.type;
        const isCheckbox = type === 'checkbox';
        const isSelect = el.tagName.toLowerCase() === 'select';

        if (el.dataset.originalValue !== undefined) {
            // Сохраняем текущее значение как оригинальное
            el.dataset.originalValue = isCheckbox ? (el.checked ? "1" : "0") : el.value;
        }

        let container = null;
        if (el.id) {
            const label = document.querySelector(`label[for="${el.id}"]`);
            if (label) container = label.querySelector('.change-container');
        }
        if (!container) container = el.parentElement.querySelector('.change-container');
        if (container) container.remove();
    });

    const globalIndicator = document.querySelector('.global-change-indicator');
    if (globalIndicator) {
        const countSpan = globalIndicator.querySelector('.global-change-count');
        if (countSpan) countSpan.textContent = '(0)';
        globalIndicator.style.display = 'none';
    }

    if (adminPlusConfig.config.other_stsn_dev_admin_plus_generate_seo === "1") generateSeo();
};

// ======================= Подсветка вкладок с ошибками =======================
const highlightErrorTabs = () => {
    document.querySelectorAll('.nav-tabs a').forEach(tab => {
        const paneId = tab.getAttribute('href');
        const pane = document.querySelector(paneId);
        if (pane && pane.querySelector('.is-invalid')) tab.classList.add('text-danger');
        else tab.classList.remove('text-danger');
    });
};

// ======================= Инициализация полей =======================
const initFields = (inputs) => {
    inputs.forEach(el => {
        if (el.tagName.toLowerCase() === 'textarea' && adminPlusConfig.config.other_stsn_dev_admin_plus_auto_heights_textarea === "1") {
            el.style.overflowY = 'hidden';
            el.setAttribute('rows', '2');
            const lineHeight = parseFloat(window.getComputedStyle(el).lineHeight) || 16;
            el.style.minHeight = `${lineHeight * 2}px`;

            const resize = () => {
                el.style.height = 'auto';
                let newHeight = el.scrollHeight;
                const maxHeight = parseInt(adminPlusConfig.config.other_stsn_dev_admin_plus_max_heights_textarea) || 0;
                if (maxHeight > 0) newHeight = Math.min(newHeight, maxHeight);
                el.style.height = `${newHeight}px`;
            };

            el.addEventListener('input', resize);
            resize();
        }

        if (adminPlusConfig.config.other_stsn_dev_admin_plus_highlight_changes === "1") createChangeHandlerUniversal(el);

        if (adminPlusConfig.config.other_stsn_dev_admin_plus_symbols_counter === "1" && /description/i.test(el.name)) {
            const counter = document.createElement('div');
            counter.className = 'textarea-counter d-inline-block align-self-center small text-muted mt-1 mx-2';
            const label = el.id ? document.querySelector(`label[for="${el.id}"]`) : null;
            if (label) label.appendChild(counter);
            else el.parentElement.insertBefore(counter, el.parentElement.firstChild);

            const updateCounter = () => { counter.textContent = `${el.value.length} s`; };
            el.addEventListener('input', updateCounter);
            updateCounter();
        }

        if (adminPlusConfig.config.other_stsn_dev_admin_plus_generate_seo === "1") {
            el.addEventListener('input', () => { generateSeo(); });
        }
    });

    document.querySelectorAll('input[type="checkbox"]').forEach(el => {
        if (adminPlusConfig.config.other_stsn_dev_admin_plus_highlight_changes === "1") createChangeHandlerUniversal(el);
    });

    document.querySelectorAll('select').forEach(el => {
        if (adminPlusConfig.config.other_stsn_dev_admin_plus_highlight_changes === "1") createChangeHandlerUniversal(el);
    });
};

// ======================= Обновление заголовка при смене названия объекта =======================
const updateH1OnNameChange = () => {
    const nameInputName = adminPlusConfig?.routeConfig?.nameSourseFieldName;
    if (!nameInputName) return;

    const nameInput = document.querySelector(`input[name="${nameInputName}"]`);
    const h1Element = document.querySelector('h1');
    const textNode = h1Element?.childNodes[0];

    if (!nameInput || !h1Element || !textNode) return;

    nameInput.addEventListener('input', () => {
        textNode.nodeValue = nameInput.value + ' ';
    });
};


// ======================= DOMContentLoaded =======================
document.addEventListener('DOMContentLoaded', () => {
    if (typeof adminPlusConfig === 'undefined') return;
    console.log('adminPlusConfig: ', adminPlusConfig);

    // Липкий header
    if (adminPlusConfig.config.other_stsn_dev_admin_plus_sticky_header === "1") {
        const header = document.querySelector('.page-header');
        if (header) header.classList.add('z-3', 'position-sticky', 'top-0', 'bg-light');
    }

    // SEO строки
    if (adminPlusConfig.config.other_stsn_dev_admin_plus_hide_disabled_stores_seo === "1") {
        updateSeoRows();
        document.querySelectorAll(`input[name="${adminPlusConfig.routeConfig.storesFieldName}[]"]`)
            .forEach(cb => cb.addEventListener('change', updateSeoRows));
    }

    // SEO значения полей
    if (adminPlusConfig.config.other_stsn_dev_admin_plus_generate_seo === "1") generateSeo();

    // Собираем все поля для глобального индикатора
    const allFields = [
        ...document.querySelectorAll('textarea, input[type="text"], input[type="checkbox"], select')
    ];

    // Инициализация полей
    initFields(allFields);

    // Глобальный индикатор изменений
    createGlobalChangeIndicator(allFields);

    // Подмена H1 при смене названия
    if (adminPlusConfig.config.other_stsn_dev_admin_plus_replace_h === "1") updateH1OnNameChange();

    // MutationObserver для alert-success
    const alertContainer = document.querySelector('#alert');
    if (alertContainer) {
        const observer = new MutationObserver(mutations => {
            mutations.forEach(mutation => {
                mutation.addedNodes.forEach(node => {
                    if (node.nodeType === 1) {
                        if (node.matches('.alert-success')) {
                            resetChangeIndicators(allFields);
                            if (adminPlusConfig.config.other_stsn_dev_admin_plus_catalog_view_button === "1") updateCatalogLinks();
                        }
                        if (adminPlusConfig.config.other_stsn_dev_admin_plus_highlight_error_tabs === "1") highlightErrorTabs();
                    }
                });
            });
        });
        observer.observe(alertContainer, { childList: true, subtree: true });
    }

    // ---- Наблюдатель за появлением/добавлением .is-invalid ----
    if (adminPlusConfig?.config?.other_stsn_dev_admin_plus_highlight_error_tabs === "1") {
        const DEBUG = false; // включи true для отладки в консоли

        const attachInvalidHandlers = (el) => {
            if (!el || el.nodeType !== 1) return;
            // не вешаем дважды
            if (el.dataset.invalidHandlerAttached === '1') return;
            el.dataset.invalidHandlerAttached = '1';

            const handler = () => {
                const tag = el.tagName.toLowerCase();
                const type = el.type;
                let valid = false;

                // Попробуем HTML5 валидацию, если есть
                try {
                    if (typeof el.checkValidity === 'function') {
                        valid = el.checkValidity();
                    }
                } catch (e) {
                    valid = false;
                }

                // Резервная логика, если checkValidity нет или он false
                if (!valid) {
                    if (tag === 'select') valid = !!el.value;
                    else if (type === 'checkbox' || type === 'radio') valid = el.checked;
                    else valid = (el.value || '').trim() !== '';
                }

                if (valid) {
                    if (DEBUG) console.debug('invalid -> fixed for', el);
                    el.classList.remove('is-invalid');
                    // Перезапускаем подсветку вкладок
                    try { highlightErrorTabs(); } catch (e) { /* silent */ }
                }
            };

            el.addEventListener('input', handler);
            el.addEventListener('change', handler);
            if (DEBUG) console.debug('attached invalid handler to', el);
        };

        const scanNodeForInvalids = (node) => {
            if (!node || node.nodeType !== 1) return;

            // Если класс поставлен на сам элемент
            if (node.classList && node.classList.contains('is-invalid')) {
                if (node.matches('input,textarea,select')) {
                    attachInvalidHandlers(node);
                } else {
                    // класс на контейнере — навешиваем на вложенные контролы
                    node.querySelectorAll && node.querySelectorAll('input,textarea,select').forEach(attachInvalidHandlers);
                }
                // обновить вкладки (на случай, если новый invalid появился)
                try { highlightErrorTabs(); } catch (e) {}
            }

            // Если добавлен контейнер — внутри могут быть контролы с .is-invalid
            if (node.querySelectorAll) {
                node.querySelectorAll('.is-invalid').forEach(found => {
                    if (found.matches('input,textarea,select')) attachInvalidHandlers(found);
                    else found.querySelectorAll('input,textarea,select').forEach(attachInvalidHandlers);
                });
            }
        };

        // Разовый скан (на всякий случай, хотя в твоём случае обычно пусто)
        scanNodeForInvalids(document.body);

        // Наблюдаем за изменением класса и за добавлением узлов
        const observerInvalid = new MutationObserver(mutations => {
            mutations.forEach(m => {
                if (m.type === 'attributes' && m.attributeName === 'class') {
                    scanNodeForInvalids(m.target);
                } else if (m.type === 'childList') {
                    m.addedNodes.forEach(n => scanNodeForInvalids(n));
                }
            });
        });

        observerInvalid.observe(document.body, {
            childList: true,
            subtree: true,
            attributes: true,
            attributeFilter: ['class']
        });

        // Удобная ручная команда для отладки в консоли:
        // window.adminPlusScanInvalids && window.adminPlusScanInvalids();
        window.adminPlusScanInvalids = () => scanNodeForInvalids(document.body);
    }
});
