<?php
declare(strict_types=1);

namespace Opencart\Admin\Controller\Extension\StsnDevAdminPlus\Other;

require_once DIR_EXTENSION . 'stsn_dev_admin_plus/helper.php';
use Opencart\Extension\StsnDevAdminPlus\StsnDevHelper as ext;

class StsnDevAdminPlus extends \Opencart\System\Engine\Controller
{
    const SLUGS = [
        'ru' => ['lang' => '/ru', 'ext' => 'smart-admin-ux-plus'],
        'uk' => ['lang' => '', 'ext' => 'smart-admin-ux-plus'],
        'en' => ['lang' => '/en', 'ext' => 'smart-admin-ux-plus'],
    ];

    const OBJ_NAMETYPES_VARIANTS = [
        'name',
        'meta_title',
    ];


    private ext $helper;
    private object $extModelsClass;
    private readonly string $separator;
    private readonly array $extInstallData;
    private readonly string $storeUrl;
    private readonly array $languagesInfo;
    private readonly array $config_keys;

    public function __construct(\Opencart\System\Engine\Registry $registry)
    {
        parent::__construct($registry);

        $this->helper = new ext($this);

        $this->load->model(ext::FULL_PATH);

        $this->extModelsClass = $this->{ext::MODEL_ALIAS};
        $this->extInstallData = $this->extModelsClass->getExtensionInstallData(ext::CODE);
        $this->storeUrl = defined('HTTPS_CATALOG') ? HTTPS_CATALOG : (defined('HTTP_CATALOG') ? HTTP_CATALOG : '');
        $this->languagesInfo = $this->model_localisation_language->getLanguages();

        $this->load->language(ext::FULL_PATH);
        $this->config_keys = [
            'status' => [
                'default' => 0,
                'label' => $this->language->get('entry_status'),
                'type' => 'checkbox'
            ],
            'license' => [
                'default' => '',
                'label' => $this->language->get('entry_license'),
                'placeholder' => $this->language->get('placeholder_license'),
                'type' => 'text'
            ],
            'collapse_left_menu' => [
                'default' => 1,
                'label' => $this->language->get('entry_collapse_left_menu'),
                'type' => 'checkbox'
            ],
            'replace_h' => [
                'default' => 1,
                'label' => $this->language->get('entry_replace_h'),
                'type' => 'checkbox'
            ],
            'catalog_view_button' => [
                'default' => 1,
                'label' => $this->language->get('entry_catalog_view_button'),
                'type' => 'checkbox'
            ],
            'catalog_view_use_admin_languaage' => [
                'default' => 1,
                'label' => $this->language->get('entry_catalog_view_use_admin_languaage'),
                'type' => 'checkbox'
            ],
            'sticky_header' => [
                'default' => 1,
                'label' => $this->language->get('entry_sticky_header'),
                'type' => 'checkbox'
            ],
            'hide_disabled_stores_seo' => [
                'default' => 1,
                'label' => $this->language->get('entry_hide_disabled_stores_seo'),
                'type' => 'checkbox',
                'child' => 'hide_stores_type'
            ],
            'hide_stores_type' => [
                'default' => 1,
                'label' => $this->language->get('entry_hide_stores_type'),
                'type' => 'radio',
                'parent' => 'hide_disabled_stores_seo',
                'options' => [
                    '1' => [
                        'text' => $this->language->get('text_hide_stores_type_not_required'),
                        'tooltip' => $this->language->get('help_hide_stores_type_not_required')
                    ],
                    '2' => [
                        'text' => $this->language->get('text_hide_stores_type_disabled'),
                        'tooltip' => $this->language->get('help_hide_stores_type_disabled')
                    ],
                ]
            ],
            'hide_disabled_languages' => [
                'default' => 1,
                'label' => $this->language->get('entry_hide_disabled_languages'),
                'type' => 'checkbox'
            ],
            'auto_heights_textarea' => [
                'default' => 1,
                'label' => $this->language->get('entry_auto_heights_textarea'),
                'type' => 'checkbox',
                'child' => 'max_heights_textarea'
            ],
            'max_heights_textarea' => [
                'default' => 300,
                'label' => $this->language->get('entry_max_heights_textarea'),
                'placeholder' => '',
                'type' => 'number',
                'min' => 0,
                'parent' => 'auto_heights_textarea'
            ],
            'symbols_counter' => [
                'default' => 1,
                'label' => $this->language->get('entry_symbols_counter'),
                'type' => 'checkbox'
            ],
            'highlight_changes' => [
                'default' => 1,
                'label' => $this->language->get('entry_highlight_changes'),
                'type' => 'checkbox'
            ],
            'highlight_error_tabs' => [
                'default' => 1,
                'label' => $this->language->get('entry_highlight_error_tabs'),
                'type' => 'checkbox'
            ],
            'generate_seo' => [
                'default' => 1,
                'label' => $this->language->get('entry_generate_seo'),
                'type' => 'checkbox'
            ],
            'extensions_submenu' => [
                'default' => 1,
                'label' => $this->language->get('entry_extensions_submenu'),
                'type' => 'checkbox'
            ],
            'draggable_layouts' => [
                'default' => 1,
                'label' => $this->language->get('entry_draggable_layouts'),
                'type' => 'checkbox'
            ],
            'unique_module_selection' => [
                'default' => 1,
                'label' => $this->language->get('entry_unique_module_selection'),
                'type' => 'checkbox'
            ],
        ];
    }

    private function formatLocalizedDate(\DateTime $date): string {
        $timezone = $this->config->get('config_timezone') ?? 'UTC';
        $date->setTimezone(new \DateTimeZone($timezone));

        // Получаем формат даты из текущего языка
        $format = $this->language->get('datetime_format') ?? 'Y-m-d H:i:s';

        return $date->format($format);
    }

    /**
     * Проверяет системные требования (наличие ionCube Loader и версия PHP).
     *
     * @return array {
     *     Результат проверки системы.
     *
     *     @type bool   'is_valid' Флаг валидности системы (true — если ошибок нет).
     *     @type string 'error'    Сообщение с описанием ошибок (пустая строка, если ошибок нет).
     * }
     */
    private function check_system(): array
    {
        $errors = [];

        if (!extension_loaded('ionCube Loader')) {
            $errors[] = $this->language->get('error_ion_cube');
        }

        if (version_compare(PHP_VERSION, ext::MIN_PHP_VERSION, '<')) {
            $errors[] = sprintf(
                $this->language->get('error_php'),
                ext::MIN_PHP_VERSION,
                PHP_VERSION
            );
        }

        return [
            'is_valid' => empty($errors),
            'error'    => implode(' ', $errors)
        ];
    }


    /**
     * @param array<int, string> $licenses  store_id => license_key
     * @param bool is_ajax
     * @return array
     */
    private function checkLicense(string $license): array
    {
        $store_url = $this->storeUrl;

        $host = parse_url($store_url, PHP_URL_HOST) ?: '';

        // Локальный хост или домен .local: лицензия не требуется
        if (str_contains($host, 'localhost') || str_ends_with($host, '.local')) {
            return [
                'is_valid'   => true,
                'hide_license' => true,
                'description' => $this->language->get('text_licensed_not_needed')
            ];
        }

        if (!$license) {
            return [
                'is_valid'   => false,
                'description' => $this->language->get('error_license_empty')
            ];
        }

        // Проверка длины лицензии: должна быть строкой длины 64 (или пустая)
        if (!is_string($license) || !in_array(strlen($license), [0, 64], true)) {
            return [
                'is_valid'   => false,
                'description' => $this->language->get('error_license_length')
            ];
        }

        $secretKeyFile = DIR_EXTENSION . ext::CODE . '/secret_key.php';
        if (file_exists($secretKeyFile)) {
            try {
                include_once $secretKeyFile;
                // Проверка статической лицензии (SHA-256 с секретным ключом, предварительно удеждаемся что secret_key.php раскодировался)
                if (\Opencart\Extension\StsnDevAdminPlus\SecretKey::verifyPermanentLicense($license, $host)) {
                    return [
                        'is_valid'   => true,
                        'hide_license' => true,
                        'description' => $this->language->get('text_licensed')
                    ];
                }
            } catch (\Throwable $e) {
                $message = 'License check failed: ' . $e->getMessage();
            }
        }

        // Удалённая проверка лицензии/триала
        $extensionCheckUrl = $this->extInstallData['link'] . '/extensions/check-trial/';
        $requestData = [
            'license'        => $license,
            'extension_code' => ext::CODE,
        ];

        $curl = curl_init($extensionCheckUrl);
        curl_setopt_array($curl, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => http_build_query($requestData),
            CURLOPT_TIMEOUT        => 5,
            CURLOPT_HTTPHEADER     => [
                'X-Requested-With: XMLHttpRequest',
                'Accept: application/json',
                'User-Agent: Mozilla/5.0 (compatible; PHP-cURL/1.0)'
            ],
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        if ($httpCode === 200 && $response !== false) {
            $serverJson = json_decode($response, true);
        } else {
            $serverJson = null;
        }

        if (is_array($serverJson) && isset($serverJson['check_result'])) {
            switch ($serverJson['check_result']) {
                case 'not-found':
                    return [
                        'is_valid'   => false,
                        'description' => $this->language->get('error_trial_not_found')
                    ];

                case 'activated':
                    $expire_d_obj = new \DateTime($serverJson['expire']);
                    return [
                        'is_valid'   => true,
                        'hide_license' => false,
                        'description' => $this->language->get('trial_activated') . $serverJson['expire']
                    ];


                case 'found':
                    $expire = new \DateTime($serverJson['expire']);
                    $now = new \DateTime('now', new \DateTimeZone('UTC'));

                    if ($now < $expire) {
                        return [
                            'is_valid'   => true,
                            'hide_license' => false,
                            'description' => $this->language->get('use_trial_valid_to') . $this->formatLocalizedDate($expire)
                        ];
                    } else {
                        return [
                            'is_valid'   => false,
                            'description' => $this->language->get('error_trial_expire')
                        ];
                    }

                default:
                    return [
                        'is_valid'   => false,
                        'description' => $this->language->get('error_invalid_server_response')
                    ];
            }
        } else {
            // Запрос не удался или ответ неверный
            return [
                'is_valid'   => false,
                'description' => $this->language->get('error_license_permanent_server_unreachable')
            ];
        }
    }

    /**
     * Получает последнюю доступную версию расширения с удалённого сервера.
     *
     * Делает POST-запрос к URL, указанному в $this->extInstallData['link'],
     * добавляя к нему путь 'extensions/get-last-version/' и передавая код расширения.
     *
     * @return string|null Возвращает строку с версией (например, '1.2.3') при успешном запросе,
     *                     или null в случае ошибки или недоступности данных.
     */
    private function getLastVersion() {
        $extensionCheckUrl = $this->extInstallData['link'] . '/extensions/get-last-version/';

        $requestData = [
            'extension_code' => ext::CODE,
            'language_code' => $this->language->get('code')
        ];

        $curl = curl_init($extensionCheckUrl);
        curl_setopt_array($curl, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => http_build_query($requestData),
            CURLOPT_TIMEOUT        => 5,
            CURLOPT_HTTPHEADER     => [
                'X-Requested-With: XMLHttpRequest',
                'Accept: application/json',
                'User-Agent: Mozilla/5.0 (compatible; PHP-cURL/1.0)'
            ],
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        if ($httpCode === 200 && $response !== false) {
            $serverJson = json_decode($response, true);
            return $serverJson;
        } else {
            $serverJson = null;
        }
        return $serverJson;
    }


    protected function makeUrl(string $route, array $params = []): string {
        $user_token = $this->session->data['user_token'];
        return $this->url->link($route, http_build_query(['user_token' => $user_token] + $params));
    }


    /**
     * @return null
     */
    public function index(): void {

        $this->document->setTitle($this->extInstallData['name']);

        $extDevSiteSlugs = self::SLUGS[$this->language->get('code')] ?? self::SLUGS['en'];
        $current_version = $this->extInstallData['version'];
        $last_version = $this->getLastVersion();

        $update_version = null;

        if ($last_version !== null && version_compare($last_version['version'], $current_version, '>')) {
            $update_version = $last_version;
        }
        $check_system_data = $this->check_system();

        $data = [
            'breadcrumbs' => [
                [
                    'text' => $this->language->get('text_home'),
                    'href' => $this->makeUrl('common/dashboard'),
                ],
                [
                    'text' => $this->language->get('text_extension'),
                    'href' => $this->makeUrl('marketplace/extension', ['type' => ext::TYPE]),
                ],
                [
                    'text' => $this->language->get('heading_title'),
                    'href' => $this->makeUrl(ext::FULL_PATH),
                ],
            ],

            'back' => $this->makeUrl('marketplace/extension', ['type' => ext::TYPE]),
            'text_edit' => sprintf($this->language->get('text_edit'), $this->extInstallData['name']),

            'header' => $this->load->controller('common/header'),
            'column_left' => $this->load->controller('common/column_left'),
            'footer' => $this->load->controller('common/footer'),

            'logo' => HTTP_CATALOG . 'extension/' . ext::CODE . '/admin/view/image/stsn-dev.svg',
            'extension_name' => $this->extInstallData['name'],
            'extension_home_page'=>  $this->extInstallData['link'] . $extDevSiteSlugs['lang'] . '/extensions/' . $extDevSiteSlugs['ext'],
            'extension_version' => $current_version,
            'update_version' => $update_version,

            'check_system_data' => $check_system_data,
            'export_url' => $this->makeUrl(ext::FULL_PATH . $this->helper->separator . 'export'),
            'import_url' => $this->makeUrl(ext::FULL_PATH . $this->helper->separator . 'import'),
        ];

        if (isset($this->session->data['success'])) {
            $data['success'] = $this->session->data['success'];
            unset($this->session->data['success']);
        } else {
            $data['success'] = '';
        }

        if (isset($this->session->data['error'])) {
            $data['error'] = $this->session->data['error'];
            unset($this->session->data['error']);
        } else {
            $data['error'] = '';
        }

        // Добавим только если check_system_data['is_valid'] === true
        if (!empty($check_system_data['is_valid'])) {
            // Конфигурационные ключи

            foreach (array_keys($this->config_keys) as $key) {
                $value = $this->helper->getExtConfigValue($key);
                $data[$key] = $value;
            }

            $data += [
                'save' => $this->makeUrl(ext::FULL_PATH . $this->helper->separator . 'save'),
                'entry_status' => sprintf($this->language->get('entry_status'), $this->extInstallData['name']),
                'license_data' => $this->checkLicense($this->helper->getExtConfigValue('license')),
                'config_fields' => $this->config_keys,
            ];
        }
        $this->response->setOutput($this->load->view(ext::FULL_PATH, $data));
    }

    public function save(): void {

        $this->load->model('setting/startup');

        $replaceFormValues = [];

        if ($this->user->hasPermission('modify', ext::FULL_PATH)) {
            if (isset($this->request->post['license'])) {
                $licenseData = $this->checkLicense($this->request->post['license']);

                $JsonResponseData = $licenseData['is_valid']
                    ? [
                        'message' => ['license' => $licenseData['description']],
                        'hide_license' =>  $licenseData['hide_license'],
                        'licensed' => true
                    ]
                    : ['error' =>['license' => $licenseData['description'], 'licensed' => false]];

                // ====================================== SAVE =============================================

                if (empty($JsonResponseData['error']['warning'])) {

                    $this->load->model('setting/event');

                    if (isset($this->request->post['status']) && $this->request->post['status'] === '1') {
                        $this->helper->save();

                        $this->model_setting_event->editStatusByCode(ext::CODE . '_view_before', true);
                        $this->model_setting_event->editStatusByCode(ext::CODE . '_view_after', true);
                    } else {
                        $this->helper->editExtConfigValue('license', $this->request->post['license']);
                        $this->helper->editExtConfigValue( 'status', 0);
                        $this->model_setting_event->editStatusByCode(ext::CODE . '_view_before', false);
                        $this->model_setting_event->editStatusByCode(ext::CODE . '_view_after', false);
                    }

                    $JsonResponseData['success'] = sprintf($this->language->get('text_success'), $this->extInstallData['name']);
                    $JsonResponseData += $replaceFormValues;
                }
            } else {
                $JsonResponseData = [];
            }
        } else {
            $JsonResponseData['error']['warning'] = $this->language->get('error_permission');
        }

        $this->response->addHeader('Content-Type: application/json');
        $this->response->setOutput(json_encode($JsonResponseData));
    }

    public function install(): void {

        $this->load->model('setting/event');

        foreach ($this->helper->events as $event) {
            // Проверяем, существует ли событие с таким кодом
            if (!$this->model_setting_event->getEventByCode($event['code'])) {
                $this->model_setting_event->addEvent(array_merge([
                    'description'=> 'customize_admin',
                    'status'     => false,
                    'sort_order' => 0
                ], $event));
            }
        }


        $this->load->model('setting/setting');

        $initData = [];

        foreach ($this->config_keys as $key => $config) {
            $initData[ext::CONF_PREFIX . '_' . $key] = $config['default'];
        }

        $this->model_setting_setting->editSetting(ext::CONF_PREFIX, $initData);
    }

    public function uninstall(): void {
        $this->load->model('setting/event');
        foreach ($this->helper->events as $event) {
            $this->model_setting_event->deleteEventByCode($event['code']);
        }
    }

    public function export() {
        if (!$this->user->hasPermission('modify', ext::FULL_PATH)) {
            return;
        }

        // Получаем все настройки по коду
        $settings = $this->db->query("SELECT * FROM `" . DB_PREFIX . "setting` WHERE `code` = '" . ext::CONF_PREFIX . "'")->rows;

        // Убираем setting_id из каждого элемента
        foreach ($settings as &$setting) {
            unset($setting['setting_id']);
        }
        unset($setting); // Очистка ссылки

        // Формируем имя файла с текущей датой и временем
        $datetime = date('Y-m-d_H-i-s');
        $filename = ext::CODE . '_settings_' . $datetime . '.json';

        header('Content-Type: application/json; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        echo json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit;
    }


    public function import() {
        if (!$this->user->hasPermission('modify', ext::FULL_PATH)) {
            return;
        }

        if (!empty($_FILES['settings']['tmp_name'])) {
            $filename = $_FILES['settings']['name'];

            if (strpos($filename, ext::CODE . '_settings_') === 0) {
                $json = json_decode(file_get_contents($_FILES['settings']['tmp_name']), true);

                if (is_array($json)) {
                    foreach ($json as $setting) {
                        $store_id = (int)$setting['store_id'];
                        $code = $this->db->escape($setting['code']);
                        $key = $this->db->escape($setting['key']);
                        $value = $this->db->escape($setting['value']);
                        $serialized = (int)$setting['serialized'];

                        $exists = $this->db->query("SELECT 1 FROM `" . DB_PREFIX . "setting` WHERE `store_id` = '$store_id' AND `code` = '$code' AND `key` = '$key' LIMIT 1");

                        if ($exists->num_rows) {
                            // Обновляем существующую запись
                            $this->db->query("UPDATE `" . DB_PREFIX . "setting` SET
                            `value` = '$value',
                            `serialized` = '$serialized'
                            WHERE `store_id` = '$store_id' AND `code` = '$code' AND `key` = '$key'
                        ");
                        } else {
                            // Вставляем новую
                            $this->db->query("INSERT INTO `" . DB_PREFIX . "setting` SET
                            `store_id` = '$store_id',
                            `code` = '$code',
                            `key` = '$key',
                            `value` = '$value',
                            `serialized` = '$serialized'
                        ");
                        }
                    }
                    $this->session->data['success'] = $this->language->get('success_import');
                } else {
                    $this->session->data['error'] = $this->language->get('error_file_json');
                }
            } else {
                $this->session->data['error'] = sprintf($this->language->get('error_file_prefix'), ext::CODE);;
            }
        } else {
            $this->session->data['error'] = $this->language->get('error_file_empty');
        }

        $url = $this->makeUrl(ext::FULL_PATH);

        if (!empty($this->session->data['error'])) {
            $this->log->write('error: ' . print_r($this->session->data['error'], true));
            $url .= '#tab-export-import';
        }

        $this->response->redirect($url);
    }
}
